/*
 i-net software provides programming examples for illustration only, without warranty
 either expressed or implied, including, but not limited to, the implied warranties
 of merchantability and/or fitness for a particular purpose. This programming example
 assumes that you are familiar with the programming language being demonstrated and
 the tools used to create and debug procedures. i-net software support professionals
 can help explain the functionality of a particular procedure, but they will not modify
 these examples to provide added functionality or construct procedures to meet your
 specific needs.
  
  i-net software 1998-2013

*/
using System;
using System.Text;
using System.Text.RegularExpressions;
using Inet.Viewer.Data;

namespace Inet.Viewer
{
    /// <summary>
    /// An exception wrapper class for the i-net Clear Reports Viewer.<br/>
    /// Contains all information necessary for dealing with various exceptions.<p/>
    /// 
    /// One of the most important uses of ViewerException is for prompt event handling:
    /// A ViewerException can be thrown if prompts are needed, in which case the flag
    /// "needPrompts" is true.
    /// </summary> 
    [Serializable]
    public class ViewerException : Exception, IHasErrorCode
    {
        private int errorCode;
        private string format;
        private string srvJVM;
        private string srvOS;
        private int srvCache;
        private PromptData[] prompts;

        /// <summary>
        /// Whether or not this is a "NeedPrompts" exception, meaning that prompt values
        /// are required by the server. If so, see the field prompts. 
        /// </summary>
        private bool needPrompts;

        /// <summary>
        /// Gets or sets the server version
        /// </summary>
        public string ServerVersion { get; private set; }

        /// <summary>
        /// Gets or sets the stacktrace on the server
        /// </summary>
        public string ServerStackTrace { get; set; }

        /// <summary>
        /// Gets or sets a flag indicating that this exception was thrown because of a cancelation.
        /// </summary>
        public bool Canceled { get; set; }

        /// <summary>
        /// Returns the error code of the exception that was thrown on report server.
        /// </summary>
        /// <returns> errorCode Error code of exception thrown on server side</returns>
        public virtual int ErrorCode
        {
            get
            {
                return errorCode;
            }
        }

        /// <summary>
        /// Gets or sets the array of prompt information - prompt information is stored as Prompt data objects 
        /// </summary>
        public virtual PromptData[] Prompts
        {
            get
            {
                return prompts;
            }
            set
            {
                this.prompts = value;
            }
        }

        /// <summary>
        /// Sets whether this exception encapsulates a prompt request - in other words meaning the report can not be rendered
        /// until the prompts specified by getPrompts() are provided to the server </summary>
        /// <param> Does this exception encapsulate a prompt request?</param>
        public virtual bool NeedPrompts
        {
            get
            {
                return needPrompts;
            }
            set
            {
                this.needPrompts = value;
            }
        }

        /// <summary>
        /// <inheritdoc/>
        /// </summary>
        public override string StackTrace
        {
            get
            {
                return ServerStackTrace + "\n\n" + base.StackTrace;
            }
        }

        /// <summary>
        /// Initializes an exception from the server with the given error code, message, export format,
        /// server version, server JAva VM, server OS, server cache version, and stacktrace. </summary>
        /// <param name="errorCode"> Error code of exception thrown on server side </param>
        /// <param name="message"> Error message of exception </param>
        /// <param name="format"> Export format at the time of the exception </param>
        /// <param name="srvVersion"> Server's i-net Clear Reports version </param>
        /// <param name="srvJVM"> Server's Java VM version </param>
        /// <param name="srvOS"> Server's Operating System </param>
        /// <param name="srvCache"> Server's Cache Version of i-net Clear Reports </param>
        /// <param name="stacktrace"> Stacktrace of exception thrown on server side.</param>
        public ViewerException(int errorCode, string message, string format, string srvVersion, string srvJVM, string srvOS, int srvCache, string stacktrace)
            : base(message)
        {
            this.errorCode = errorCode;
            this.format = format;
            this.ServerVersion = srvVersion;
            this.srvJVM = srvJVM;
            this.srvOS = srvOS;
            this.srvCache = srvCache;
            this.ServerStackTrace = stacktrace;
        }

        /// <summary>
        /// Initializes an exception with the given error message </summary>
        /// <param name="message"> Error message</param>
        public ViewerException(string message)
            : base(message)
        {
        }

        /// <summary>
        /// Initializes an exception with the given error message and wraps another Throwable in this ViewerException </summary>
        /// <param name="message"> Error message of this exception </param>
        /// <param name="cause">exception to wrap in this ViewerException</param>
        public ViewerException(string message, Exception cause)
            : base(message, cause)
        {
        }

        /// <summary>
        /// For the "need prompts" Viewer Exception </summary>
        /// <param name="prompts"> Prompts needed: Array of prompt information - each prompt data object
        /// holds information about what type of prompt, what default values there are, etc. </param>
        /// <seealso cref= "PromptData"/>
        public ViewerException(PromptData[] prompts)
            : base("NeedsPrompts")
        {
            this.prompts = prompts;
            needPrompts = true;
            ViewerUtils.Debug("Creating Viewer Exception - PROMPTS!");
        }

        /// <summary>
        /// Simple factory method for creating a ViewerException as a Wrapper for a Throwable. </summary>
        /// <param name="cause">exception to wrap in this ViewerException </param>
        /// <returns> New ViewerException wrapping Throwable </returns>
        public static ViewerException CreateViewerException(Exception cause)
        {
            if (cause is ViewerException)
            {
                return (ViewerException)cause;
            }
            return new ViewerException(GetErrorMessage(cause), cause);
        }

        /// <summary>
        /// Wraps the given exception in a ViewerException. If the given exception is a ViewerException, the message is
        /// appended to the beginning and all parameters such as error code, etc., are preserved. </summary>
        /// <param name="message"> String to append to the beginning </param>
        /// <param name="cause"> Throwable to wrap </param>
        /// <returns> New ViewerException wrapping Throwable with appended message</returns>
        public static ViewerException CreateViewerExceptionWithMessage(string message, Exception cause)
        {
            if (cause is ViewerException)
            {
                ViewerException ve = (ViewerException)cause;
                return new ViewerException(ve.errorCode, message + "\r\n" + ve.Message, ve.format, ve.ServerVersion, ve.srvJVM, ve.srvOS, ve.srvCache, ve.ServerStackTrace);
            }
            else
            {
                return new ViewerException(message, cause);
            }
        }

        /// <summary>
        /// Create a user friendly error message. Returns the error message contained in the given throwable, add the class name without package if it to short. </summary>
        /// <param name="th">exception to extract error message from, truncated if necessary </param>
        /// <returns> Message contained in the Throwable.</returns>
        public static string GetErrorMessage(Exception th)
        {
            string msg = th.Message;
            /*
             * Add classname without package, if:
             * - Message is null
             * - very short (smaller 20)
             * - only one word (No empty spaces)
             */
            if (msg == null || msg.Length < 20 || msg.IndexOf(' ') < 0)
            {
                string className = th.GetType().Name;
                className = className.Substring(className.LastIndexOf('.') + 1);
                if (th is ViewerException || className.Equals("ReportException"))
                {
                    // Our own Exceptionss should already have a userfriendly error message
                    if (msg == null)
                    {
                        // msg can't be null
                        msg = string.Empty;
                    }
                    return msg;
                }
                StringBuilder builder = new StringBuilder();
                string[] classWords = Regex.Split(className, "(?<=[a-z])(?=[A-Z])");
                foreach (string word in classWords)
                {
                    if (word.Equals("Exception") && builder.Length > 0)
                    {
                        continue;
                    }
                    if (builder.Length > 0)
                    {
                        builder.Append(" ");
                    }
                    builder.Append(word);
                }
                if (msg != null)
                {
                    builder.Append(": ").Append(msg);
                }
                return builder.ToString();
            }
            return msg;
        }
    }
}